#!/usr/bin/env python3

# This script converts https://amrdc.ssec.wisc.edu netCDF polar composite satellite imagery
# to Cloud Optimized GeoTIFFs.
#
# Usage:
#   ./PolarComposite2GeoTIFF.py <input file>
#
# Example:
#   ./PolarComposite2GeoTIFF.py Antarctic.Composite.4km.WaterVapor.2025.11.19.0100Z.nc.gz
#
# 2025-11-19 First version Sam Batzli (SAB)
# 2025-12-10 v1.1 SAB Added absolute path variable
# 2025-12-22 v1.2 SAB Added logic to find gdal tools
# 2026-01-04 v1.3 SAB Added PATH check before using find

import subprocess
import argparse
import os
import gzip
import shutil
import sys


def find_executable(name):
    """Find an executable in PATH or via find."""
    path = shutil.which(name)
    if path:
        return path

    try:
        result = subprocess.run(
            ["find", "/", "-type", "f", "-name", name],
            stdout=subprocess.PIPE,
            stderr=subprocess.DEVNULL,
            text=True,
            check=True,
        )
        matches = result.stdout.strip().splitlines()
        if matches:
            return matches[0]
    except subprocess.CalledProcessError:
        pass

    return None


def main(input_file):
    # Locate GDAL tools
    gdal_translate = find_executable("gdal_translate")
    gdalwarp = find_executable("gdalwarp")

    if not gdal_translate:
        print("ERROR: gdal_translate not found.", file=sys.stderr)
        sys.exit(1)

    if not gdalwarp:
        print("ERROR: gdalwarp not found.", file=sys.stderr)
        sys.exit(1)

    print(f"Using gdal_translate at: {gdal_translate}")
    print(f"Using gdalwarp at: {gdalwarp}")

    uncompressed_file = None

    # Handle gzipped input
    if input_file.endswith(".gz"):
        uncompressed_file = input_file[:-3]
        with gzip.open(input_file, "rb") as f_in, open(uncompressed_file, "wb") as f_out:
            shutil.copyfileobj(f_in, f_out)

        print(f"Uncompressed: {input_file} -> {uncompressed_file}")
        input_file = uncompressed_file

    current_directory = os.getcwd()
    print("Current Working Directory:", current_directory)

    base_name = os.path.splitext(input_file)[0]
    output_tif = os.path.join(current_directory, f"{base_name}.tif")
    output2_tif = os.path.join(current_directory, f"{base_name}.3031.tif")

    lat_vrt = os.path.join(current_directory, f"{base_name}.lat.vrt")
    lon_vrt = os.path.join(current_directory, f"{base_name}.lon.vrt")
    data_vrt = os.path.join(current_directory, f"{base_name}.data.vrt")

    # Projection selection
    if input_file.startswith("Arctic.Composite"):
        proj_string = (
            "+proj=stere +lat_0=90 +lat_ts=60 +lon_0=-170 "
            "+x_0=0 +y_0=0 +datum=WGS84 +units=m +no_defs"
        )
    elif input_file.startswith("Antarctic.Composite"):
        proj_string = (
            "+proj=stere +lat_0=-90 +lat_ts=-60 +lon_0=-140 "
            "+x_0=0 +y_0=0 +datum=WGS84 +units=m +no_defs"
        )
    else:
        print("ERROR: Unsupported input file naming convention.", file=sys.stderr)
        sys.exit(1)

    commands = [
        [
            gdal_translate,
            "-of", "VRT",
            f'NETCDF:"{os.path.join(current_directory, input_file)}":lat',
            lat_vrt,
        ],
        [
            gdal_translate,
            "-of", "VRT",
            f'NETCDF:"{os.path.join(current_directory, input_file)}":lon',
            lon_vrt,
        ],
        [
            gdal_translate,
            "-of", "VRT",
            f'NETCDF:"{os.path.join(current_directory, input_file)}":data',
            data_vrt,
        ],
        [
            gdalwarp,
            "-geoloc",
            "-of", "COG",
            "-co", "COMPRESS=LZW",
            "-co", "BIGTIFF=YES",
            "-co", "BLOCKSIZE=512",
            "-co", "OVERVIEWS=AUTO",
            "-co", "PREDICTOR=YES",
            "-t_srs", proj_string,
            "-r", "nearest",
            data_vrt,
            output_tif,
        ],
        [
            gdalwarp,
            "-t_srs", "EPSG:3031",
            "-r", "nearest",
            output_tif,
            output2_tif,
        ],
    ]

    for command in commands:
        try:
            result = subprocess.run(
                command, check=True, text=True, capture_output=True
            )
            print(f"Command succeeded: {' '.join(command)}")
        except subprocess.CalledProcessError as e:
            print(f"ERROR running: {' '.join(command)}", file=sys.stderr)
            print(e.stderr, file=sys.stderr)
            sys.exit(1)

    for vrt in (lat_vrt, lon_vrt, data_vrt):
        try:
            os.remove(vrt)
            print(f"Deleted: {vrt}")
        except FileNotFoundError:
            pass

    if uncompressed_file:
        try:
            os.remove(uncompressed_file)
            print(f"Deleted temporary file: {uncompressed_file}")
        except FileNotFoundError:
            pass


if __name__ == "__main__":
    parser = argparse.ArgumentParser(
        description="Convert polar composite NETCDF files to Cloud Optimized GeoTIFFs"
    )
    parser.add_argument(
        "input_file",
        help="Input NETCDF file (.nc or .nc.gz)",
    )

    args = parser.parse_args()
    main(args.input_file)

